<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Adds Data privacy-related settings.
 *
 * @package    tool_dataprivacy
 * @copyright  2018 onwards Jun Pataleta
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die;

if ($hassiteconfig) {
    $privacysettings = $ADMIN->locate('privacysettings');

    if ($ADMIN->fulltree) {
        // Contact data protection officer. Disabled by default.
        $privacysettings->add(new admin_setting_configcheckbox('tool_dataprivacy/contactdataprotectionofficer',
                new lang_string('contactdataprotectionofficer', 'tool_dataprivacy'),
                new lang_string('contactdataprotectionofficer_desc', 'tool_dataprivacy'), 0)
        );

        $privacysettings->add(new admin_setting_configcheckbox('tool_dataprivacy/automaticdataexportapproval',
                new lang_string('automaticdataexportapproval', 'tool_dataprivacy'),
                new lang_string('automaticdataexportapproval_desc', 'tool_dataprivacy'), 0)
        );

        $privacysettings->add(new admin_setting_configcheckbox('tool_dataprivacy/automaticdatadeletionapproval',
                new lang_string('automaticdatadeletionapproval', 'tool_dataprivacy'),
                new lang_string('automaticdatadeletionapproval_desc', 'tool_dataprivacy'), 0)
        );

        // Automatically create delete data request for users upon user deletion.
        // Automatically create delete data request for pre-existing deleted users.
        // Enabled by default.
        $privacysettings->add(new admin_setting_configcheckbox('tool_dataprivacy/automaticdeletionrequests',
                new lang_string('automaticdeletionrequests', 'tool_dataprivacy'),
                new lang_string('automaticdeletionrequests_desc', 'tool_dataprivacy'), 1)
        );

        // Set days approved data requests will be accessible. 1 week default.
        $privacysettings->add(new admin_setting_configduration('tool_dataprivacy/privacyrequestexpiry',
                new lang_string('privacyrequestexpiry', 'tool_dataprivacy'),
                new lang_string('privacyrequestexpiry_desc', 'tool_dataprivacy'),
                WEEKSECS, 1));

        // Fetch roles that are assignable.
        $assignableroles = get_assignable_roles(context_system::instance());

        // Fetch roles that have the capability to manage data requests.
        $capableroles = get_roles_with_capability('tool/dataprivacy:managedatarequests');

        // Role(s) that map to the Data Protection Officer role. These are assignable roles with the capability to
        // manage data requests.
        $roles = [];
        foreach ($capableroles as $key => $role) {
            if (array_key_exists($key, $assignableroles)) {
                $roles[$key] = $assignableroles[$key];
            }
        }
        if (!empty($roles)) {
            $privacysettings->add(new admin_setting_configmulticheckbox('tool_dataprivacy/dporoles',
                    new lang_string('dporolemapping', 'tool_dataprivacy'),
                    new lang_string('dporolemapping_desc', 'tool_dataprivacy'), null, $roles)
            );
        }

        // When calculating user expiry, should courses which have no end date be considered.
        $privacysettings->add(new admin_setting_configcheckbox('tool_dataprivacy/requireallenddatesforuserdeletion',
                new lang_string('requireallenddatesforuserdeletion', 'tool_dataprivacy'),
                new lang_string('requireallenddatesforuserdeletion_desc', 'tool_dataprivacy'),
                1));

        // Whether the data retention summary should be shown in the page footer and in the user profile page.
        $privacysettings->add(new admin_setting_configcheckbox('tool_dataprivacy/showdataretentionsummary',
            new lang_string('showdataretentionsummary', 'tool_dataprivacy'),
            new lang_string('showdataretentionsummary_desc', 'tool_dataprivacy'),
            1));

        // Whether to allow PO to select courses for data export, instead of always exporting all data.
        $privacysettings->add(new admin_setting_configcheckbox('tool_dataprivacy/allowfiltering',
            new lang_string('allowfiltering', 'tool_dataprivacy'),
            new lang_string('allowfiltering_desc', 'tool_dataprivacy'),
            0));
        // Prevent the case where the automaticdataexportapproval setting is set to automatically approve,
        // but the allowfiltering option is also enabled and non-functional.
        $privacysettings->hide_if('tool_dataprivacy/allowfiltering', 'tool_dataprivacy/automaticdataexportapproval', 'checked', 1);
    }
}

// Restrict config links to the DPO.
if (tool_dataprivacy\api::is_site_dpo($USER->id)) {
    // Link that leads to the data requests management page.
    $ADMIN->add('privacy', new admin_externalpage('datarequests', get_string('datarequests', 'tool_dataprivacy'),
        new moodle_url('/admin/tool/dataprivacy/datarequests.php'), 'tool/dataprivacy:managedatarequests')
    );

    // Link that leads to the data registry management page.
    $ADMIN->add('privacy', new admin_externalpage('dataregistry', get_string('dataregistry', 'tool_dataprivacy'),
        new moodle_url('/admin/tool/dataprivacy/dataregistry.php'), 'tool/dataprivacy:managedataregistry')
    );

    // Link that leads to the review page of expired contexts that are up for deletion.
    $ADMIN->add('privacy', new admin_externalpage('datadeletion', get_string('datadeletion', 'tool_dataprivacy'),
            new moodle_url('/admin/tool/dataprivacy/datadeletion.php'), 'tool/dataprivacy:managedataregistry')
    );

    // Link that leads to the other data registry management page.
    $ADMIN->add('privacy', new admin_externalpage('pluginregistry', get_string('pluginregistry', 'tool_dataprivacy'),
        new moodle_url('/admin/tool/dataprivacy/pluginregistry.php'), 'tool/dataprivacy:managedataregistry')
    );
}
